#include "MatchingSony.h"
#include "MessagingSony.h"
#include "UserProfileSony.h"
#include "MessagePipe.h"
#include "ErrorCodesSony.h"
#include "SignInSony.h"

using namespace sce::Toolkit::NP;
using namespace sce::Toolkit::NP::Utilities;

namespace UnityPlugin
{
	extern std::string gSessionImageFilePath;

	CachedMatching gMatching;

	DO_EXPORT( bool, PrxSessionIsBusy ) ()
	{
		return gMatching.SessionIsBusy();
	}

	DO_EXPORT( bool, PrxSessionGetLastError) (ResultCode* result)
	{
		return gMatching.GetLastError(result);
	}

	DO_EXPORT( ErrorCode, PrxSessionClearAttributeDefinitions ) ()
	{
		return gMatching.ClearAttributeDefinitions();
	}

	DO_EXPORT( ErrorCode, PrxSessionAddAttributeDefinition ) (const char* name, int type, int valueType, int maxSize)
	{
		return gMatching.AddAttributeDefinition(name, type, valueType, maxSize);
	}

	DO_EXPORT( ErrorCode, PrxSessionRegisterAttributeDefinitions) ()
	{
		return gMatching.RegisterAttributeDefinitions();
	}

	DO_EXPORT( bool, PrxSessionInSession ) ()
	{
		return gMatching.InSession();
	}

	DO_EXPORT( ErrorCode, PrxSessionClearAttributes ) ()
	{
		return gMatching.ClearSessionAttributes();
	}

	DO_EXPORT( ErrorCode, PrxSessionAddAttribute ) (SessionAttribute* attribute)
	{
		return gMatching.AddSessionAttribute(attribute);
	}

	DO_EXPORT( ErrorCode, PrxSessionCreateSession ) (const char* name, int serverID, int worldID, int numSlots, const char* password, int creationFlags, int typeFlags, const char* sessionStatus)
	{
		return gMatching.CreateSession(name, serverID, worldID, numSlots, 0, password, creationFlags, typeFlags, sessionStatus);
	}

	DO_EXPORT( ErrorCode, PrxSessionCreateFriendsSession ) (const char* name, int serverID, int worldID, int numSlots, int numFriendSlots, const char* password, int creationFlags, const char* sessionStatus)
	{
		return gMatching.CreateSession(name, serverID, worldID, numSlots, numFriendSlots, password, creationFlags, 0, sessionStatus);
	}

	DO_EXPORT( ErrorCode, PrxSessionJoinSession ) (int sessionID, const char* password)
	{
		return gMatching.JoinSession(sessionID, password);
	}

	DO_EXPORT( ErrorCode, PrxSessionJoinInvitedSession ) ()
	{
		return gMatching.JoinInvitedSession();
	}

	// Session Modification...
	DO_EXPORT( ErrorCode, PrxSessionModifyClearAttributes ) ()
	{
		return gMatching.ClearModifySessionAttributes();
	}

	DO_EXPORT( ErrorCode, PrxSessionModifyAddAttribute ) (ModifySessionAttribute* attribute)
	{
		return gMatching.AddModifySessionAttribute(attribute);
	}

	DO_EXPORT( ErrorCode, PrxSessionModifySession ) (int attributeType)
	{
		return gMatching.ModifySession(attributeType);
	}

	DO_EXPORT( ErrorCode, PrxSessionLeaveSession ) ()
	{
		return gMatching.LeaveSession();
	}

	// Finding sessions...
	DO_EXPORT( ErrorCode, PrxSessionFind ) (int serverID, int worldID)
	{
		return gMatching.Search(serverID, worldID);
	}

	DO_EXPORT( ErrorCode, PrxSessionFindFriends ) (int serverID, int worldID)
	{
		return gMatching.SearchFriends(serverID, worldID);
	}

	DO_EXPORT( ErrorCode, PrxSessionFindRegional ) (int serverID, int worldID)
	{
		return gMatching.SearchRegional(serverID, worldID);
	}

	// Found session info...
	DO_EXPORT( void, PrxLockFoundSessionList ) ()
	{
		gMatching.Lock();
	}

	DO_EXPORT( void, PrxUnlockFoundSessionList ) ()
	{
		gMatching.Unlock();
	}

	DO_EXPORT( int, PrxGetFoundSessionListCount ) ()
	{
		return gMatching.GetSessionFoundListCount();
	}

	DO_EXPORT( ErrorCode, PrxGetFoundSessionInfo ) (int sessionIndex, SessionInfo* info)
	{
		return gMatching.GetSessionFoundInfo(sessionIndex, info);
	}

	// Found session attributes
	DO_EXPORT( void, PrxLockFoundSessionAttributeList ) ()
	{
		gMatching.Lock();
	}

	DO_EXPORT( void, PrxUnlockFoundSessionAttributeList ) ()
	{
		gMatching.Unlock();
	}

	DO_EXPORT( int, PrxGetFoundSessionAttributeListCount ) (int sessionIndex)
	{
		return gMatching.GetSessionFoundAttributeCount(sessionIndex);
	}

	DO_EXPORT( ErrorCode, PrxGetFoundSessionAttributeInfo ) (int sessionIndex, int attributeIndex, SessionAttributeInfo* info)
	{
		return gMatching.GetSessionFoundAttribute(sessionIndex, attributeIndex, info);
	}

	// Active session info...
	DO_EXPORT( ErrorCode, PrxGetSessionInfo ) (SessionInfo* info)
	{
		return gMatching.GetSessionInfo(info);
	}

	// Active session attributes
	DO_EXPORT( void, PrxLockSessionAttributeList ) ()
	{
		gMatching.Lock();
	}

	DO_EXPORT( void, PrxUnlockSessionAttributeList ) ()
	{
		gMatching.Unlock();
	}

	DO_EXPORT( int, PrxGetSessionAttributeListCount ) ()
	{
		return gMatching.GetSessionAttributeCount();
	}

	DO_EXPORT( ErrorCode, PrxGetSessionAttributeInfo ) (int index, SessionAttributeInfo* info)
	{
		return gMatching.GetSessionAttribute(index, info);
	}

	// Active session member info...
	DO_EXPORT( void, PrxLockSessionMemberList ) ()
	{
		gMatching.Lock();
	}

	DO_EXPORT( void, PrxUnlockSessionMemberList ) ()
	{
		gMatching.Unlock();
	}

	DO_EXPORT( int, PrxGetSessionMemberListCount ) ()
	{
		return gMatching.GetSessionMemberCount();
	}

	DO_EXPORT( ErrorCode, PrxGetSessionMemberInfo ) (int index, SessionMemberInfo* info)
	{
		return gMatching.GetSessionMemberInfo(index, info);
	}

	// Active session member attributes...
	DO_EXPORT( void, PrxLockSessionMemberAttributeList ) ()
	{
		gMatching.Lock();
	}

	DO_EXPORT( void, PrxUnlockSessionMemberAttributeList ) ()
	{
		gMatching.Unlock();
	}

	DO_EXPORT( int, PrxGetSessionMemberAttributeListCount ) (int memberIndex)
	{
		return gMatching.GetSessionMemberAttributeCount(memberIndex);
	}

	DO_EXPORT( ErrorCode, PrxGetSessionMemberAttributeInfo ) (int memberIndex, int index, SessionAttributeInfo* info)
	{
		return gMatching.GetSessionMemberAttribute(memberIndex, index, info);
	}

	// Session invites
	DO_EXPORT( ErrorCode, PrxInviteToSession ) (const char* text, int npIdsCount)
	{
		return gMatching.InviteToSession(text, npIdsCount);
	}

	CachedMatching::CachedMatching()
		: m_SessionBusy(false)
		, m_LastResult("Matching")
	{
		memset(&m_SessionInformation, 0, sizeof(SessionInformation));
	}

	bool CachedMatching::ProcessEvent(const sce::Toolkit::NP::Event& event)
	{
		SimpleLock::AutoLock lock(m_Lock);
		bool handled = false;

		switch(event.event)
		{
			case Event::matchingSessionCreated:				// An event generated when session creation has been completed.
				if(m_FutureSessionProcess.getError() != SCE_TOOLKIT_NP_SUCCESS)
				{
					m_LastResult.SetResultSCE(m_FutureSessionProcess.getError(), true, __FUNCTION__, __LINE__);
					Messages::AddMessage(Messages::kNPToolKit_MatchingError);
				}
				else
				{
					m_SessionInformation = *m_FutureSessionProcess.get();
#if __ORBIS__ || PSP2_USING_WEBAPI
					if (m_SessionInformation.npSession.errorCode != SCE_OK)
					{
						// The session was created but SessionInformation.npSession is not valid);
						Messages::LogWarning("Session created, but npSession has error: %s", LookupSceErrorCode(m_SessionInformation.npSession.errorCode ));
					}
#endif
					Messages::AddMessage(Messages::kNPToolKit_MatchingCreatedSession);
				}

				m_SessionBusy = false;
				handled = true;
				break;

			case Event::matchingSessionJoined:				// An event generated when the join session process has been completed.
				if(m_FutureSessionProcess.getError() != SCE_TOOLKIT_NP_SUCCESS)
				{
					switch(m_FutureSessionProcess.getError())
					{
						case SCE_NP_MATCHING2_SERVER_ERROR_NO_SUCH_ROOM:
							Messages::AddMessage(Messages::kNPToolKit_MatchingJoinInvalidSession);
							break;

						default:
							m_LastResult.SetResultSCE(m_FutureSessionProcess.getError(), true, __FUNCTION__, __LINE__);
							Messages::AddMessage(Messages::kNPToolKit_MatchingError);
							break;
					}
				}
				else
				{
					m_SessionInformation = *m_FutureSessionProcess.get();
					Messages::AddMessage(Messages::kNPToolKit_MatchingJoinedSession);
				}

				m_SessionBusy = false;
				handled = true;
				break;

			case Event::matchingSessionUpdate:				// An event generated when the session has been updated. 
				{
					SceNpMatching2Event matchingEvent;
					int ret = Matching::Interface::updateSession(&m_SessionInformation, &matchingEvent, event.event );
					switch(ret)
					{
						case (int)SCE_TOOLKIT_NP_SUCCESS:
							Messages::AddMessage(Messages::kNPToolKit_MatchingUpdatedSession);
							break;

						case (int)SCE_TOOLKIT_NP_MATCHING_SESSION_KICKEDOUT:
							Messages::AddMessage(Messages::kNPToolKit_MatchingKickedOut);
							break;

						case (int)SCE_TOOLKIT_NP_MATCHING_SESSION_ROOM_DESTROYED:
							Messages::AddMessage(Messages::kNPToolKit_MatchingRoomDestroyed);
							break;

						default:
							Messages::LogWarning("matchingSessionUpdate - %s", LookupSceErrorCode(ret));
							break;
					}
				}
				handled = true;
				break;

			case Event::matchingSessionLeft:				// An event generated when the user has left the current session.
				if(m_FutureSessionResult.getError() != SCE_TOOLKIT_NP_SUCCESS)
				{
					m_LastResult.SetResultSCE(m_FutureSessionProcess.getError(), true, __FUNCTION__, __LINE__);
					Messages::AddMessage(Messages::kNPToolKit_MatchingError);
				}
				else
				{
					memset(&m_SessionInformation, 0, sizeof(SessionInformation));
					Messages::AddMessage(Messages::kNPToolKit_MatchingLeftSession);
				}
				m_FutureSessionResult.reset();
				m_SessionBusy = false;
				handled = true;
				break;

			case Event::matchingSessionSearchCompleted:		// An event generated when the search process has been completed.
				if (m_FutureSearchProcess->hasResult())
				{
					m_SessionList = *m_FutureSearchProcess->get();
					Messages::AddMessage(Messages::kNPToolKit_MatchingFoundSessions);
				}
				else if(m_FutureSearchProcess->getError() != SCE_TOOLKIT_NP_SUCCESS)
				{
					m_LastResult.SetResultSCE(m_FutureSearchProcess->getError(), true, __FUNCTION__, __LINE__);
					Messages::AddMessage(Messages::kNPToolKit_MatchingError);
				}

				DestroySessionSearchObject();
				m_SessionBusy = false;
				handled = true;
				break;

			case Event::matchingSessionModified:			// An event generated when the session has been modified.
				if(m_FutureSessionResult.getError() != SCE_TOOLKIT_NP_SUCCESS)
				{
					m_LastResult.SetResultSCE(m_FutureSessionResult.getError(), true, __FUNCTION__, __LINE__);
					Messages::AddMessage(Messages::kNPToolKit_MatchingError);
				}
				else
				{
					SceNpMatching2Event matchingEvent;
					int ret = Matching::Interface::updateSession(&m_SessionInformation, &matchingEvent, event.event );
					if(ret != SCE_TOOLKIT_NP_SUCCESS)
					{
						Messages::LogWarning("matchingSessionUpdate - %s", LookupSceErrorCode(ret));
					}
					else
					{
						Messages::AddMessage(Messages::kNPToolKit_MatchingUpdatedSession);
					}
				}
				m_FutureSessionResult.reset();
				m_SessionBusy = false;
				handled = true;
				break;

			case Event::matchingSessionError:				// An event generated when there was an error performing the current process.
			case Event::matchingSessionMessageSentError:	// An event generated when the session message has been sent.
				Messages::LogWarning("Session event not handled: event=%d\n", event.event);
				handled = true;
				break;


			default:
				break;
		}

		return handled;
	}

	ErrorCode CachedMatching::ClearAttributeDefinitions()
	{
		SimpleLock::AutoLock lock(m_Lock);
		m_LastResult.Reset();
		m_AttributeDefinitions.clear();
		return m_LastResult.GetResult();
	}

	ErrorCode CachedMatching::AddAttributeDefinition(const char* name, int type, int valueType, int maxSize)
	{
		SimpleLock::AutoLock lock(m_Lock);
		m_LastResult.Reset();
		RegisterSessionAttribute definition;
		memset(&definition, 0, sizeof(RegisterSessionAttribute));
		strncpy(definition.attribute, name, SCE_TOOLKIT_NP_MAX_ATTRIBUTE_LENGTH);
		definition.attributeType = type;
		definition.valueType = valueType;
		definition.maxSize = maxSize;
		m_AttributeDefinitions.push_back(definition);
		return m_LastResult.GetResult();
	}

	ErrorCode CachedMatching::RegisterAttributeDefinitions()
	{
		SimpleLock::AutoLock lock(m_Lock);
		
		int ret = Matching::Interface::registerSessionAttributes(&m_AttributeDefinitions[0], m_AttributeDefinitions.size());
		if (ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
			return m_LastResult.GetResult();
		}

		return m_LastResult.GetResult();
	}

	bool CachedMatching::InSession()
	{
		SimpleLock::AutoLock lock(m_Lock);
		return m_SessionInformation.numMembers > 0;
	}

	bool CachedMatching::SessionIsBusy()
	{
		SimpleLock::AutoLock lock(m_Lock);
		return m_SessionBusy;
	}

	ErrorCode CachedMatching::ClearSessionAttributes()
	{
		SimpleLock::AutoLock lock(m_Lock);

		m_LastResult.Reset();
		m_SessionAttributes.clear();
		return m_LastResult.GetResult();
	}

	ErrorCode CachedMatching::AddSessionAttribute(const SessionAttribute* attribute)
	{
		SimpleLock::AutoLock lock(m_Lock);

		m_LastResult.Reset();
		SessionRequestAttribute attrib;
		memset(&attrib, 0, sizeof(SessionRequestAttribute));
		strncpy(attrib.attribute, attribute->name, SCE_TOOLKIT_NP_MAX_ATTRIBUTE_LENGTH);
		if(attribute->binValue && strlen(attribute->binValue) > 0)
		{
			strncpy(attrib.attributeValue.attributeBinValue, attribute->binValue, SCE_TOOLKIT_NP_ATTRIBUTE_MAX_BIN_VALUE);
		}
		else
		{
			attrib.attributeValue.attributeIntValue = attribute->intValue;
		}
		attrib.searchOperator = attribute->searchOperator;

		m_SessionAttributes.push_back(attrib);
		return m_LastResult.GetResult();
	}



	ErrorCode CachedMatching::CreateSession(const char* name, int serverID, int worldID, int numSlots, int numFriendSlots, const char* password, int creationFlags, int typeFlags, const char* sessionStatus)
	{
		if(SessionIsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		SimpleLock::AutoLock lock(m_Lock);

		m_LastResult.Reset();
		CreateSessionRequest sessionRequest;
		INIT(sessionRequest);
		
#if __ORBIS__ || PSP2_USING_WEBAPI
		strncpy(sessionRequest.sessionImgPath, gSessionImageFilePath.c_str(), SCE_TOOLKIT_NP_SESSION_IMAGE_PATH_MAX_SIZE);
		strncpy(sessionRequest.sessionStatus, sessionStatus, SCE_TOOLKIT_NP_SESSION_STATUS_MAX_SIZE);
		sessionRequest.userInfo.userId = GetUserId();
#endif
		sessionRequest.numSessionAttributes = m_SessionAttributes.size();
		sessionRequest.sessionAttributes = sessionRequest.numSessionAttributes > 0 ? &m_SessionAttributes[0] : NULL;
		sessionRequest.createSessionFlag = creationFlags;
		sessionRequest.sessionTypeFlag = typeFlags;
		sessionRequest.maxSlots = numSlots;
		if(sessionRequest.sessionTypeFlag == 0)
		{
			if(numFriendSlots > numSlots)
			{
				return m_LastResult.SetResult(NP_ERR_INDEX_OUT_OF_RANGE, true, __FUNCTION__, __LINE__);
			}
			sessionRequest.slotsInformation.reservedSlots = numFriendSlots;
			sessionRequest.slotsInformation.openSlots = numSlots - numFriendSlots;
		}
		sessionRequest.serverId = serverID;
		sessionRequest.worldId	= worldID;
		if(name)
		{
			strncpy(sessionRequest.sessionName, name, SCE_TOOLKIT_NP_SESSION_NAME_MAX_SIZE);
		}
		if(password)
		{
			strncpy(sessionRequest.sessionPassword, password, SCE_NP_MATCHING2_SESSION_PASSWORD_SIZE);
		}

		int ret = Matching::Interface::createSession(&sessionRequest, &m_FutureSessionProcess);
		if (ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		m_SessionBusy = true;
		return m_LastResult.GetResult();
	}

	// Join a session from an index in the session list
	ErrorCode CachedMatching::JoinSession(unsigned int sessionID, const char* password)
	{
		if(SessionIsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		SimpleLock::AutoLock lock(m_Lock);

		m_LastResult.Reset();

		if(sessionID >= m_SessionList.size())
		{
			return m_LastResult.SetResult(NP_ERR_INDEX_OUT_OF_RANGE, true, __FUNCTION__, __LINE__);
		}

		JoinSessionRequest sessionToJoinInfo;

		memset(&sessionToJoinInfo,0, sizeof(JoinSessionRequest));
#if __ORBIS__ || PSP2_USING_WEBAPI
		sessionToJoinInfo.userInfo.userId = GetUserId();
#endif
		sessionToJoinInfo.numSessionAttributes = m_SessionAttributes.size();
		sessionToJoinInfo.memberAttributes = sessionToJoinInfo.numSessionAttributes > 0 ? &m_SessionAttributes[0] : NULL;
		sessionToJoinInfo.sessionInformation  = &m_SessionList[sessionID];

		// If a password was supplied and the session has reserved slots, set the password.
		if(password && (strlen(password) > 0) && (sessionToJoinInfo.sessionInformation->slotsInformation.reservedSlots > 0))
		{
			static SceNpMatching2SessionPassword pass;
			strncpy((char*)pass.data, password, SCE_NP_MATCHING2_SESSION_PASSWORD_SIZE);
			sessionToJoinInfo.sessionPassword = &pass;
		}

		int ret = Matching::Interface::joinSession(&sessionToJoinInfo, &m_FutureSessionProcess);
		if (ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		m_SessionBusy = true;
		return m_LastResult.GetResult();
	}


	ErrorCode CachedMatching::JoinInvitedSession()
	{

#if	NP_HAS_MESSAGING
		sce::Toolkit::NP::MessageAttachment* attachment = gMessaging.GetSessionInviteAttachment();
		if(attachment == NULL)
		{
			return m_LastResult.SetResult(NP_ERR_FAILED, true, __FUNCTION__, __LINE__);
		}
#endif
		if(SessionIsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		SimpleLock::AutoLock lock(m_Lock);

#if	NP_HAS_MESSAGING
		int ret = Matching::Interface::joinInvitedSession(attachment, &m_FutureSessionProcess);
		if (ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}
#endif
		m_SessionBusy = true;
		return m_LastResult.GetResult();
	}

	ErrorCode CachedMatching::ClearModifySessionAttributes()
	{
		SimpleLock::AutoLock lock(m_Lock);
		m_LastResult.Reset();
		m_ModifyAttributes.clear();
		return m_LastResult.GetResult();
	}

	ErrorCode CachedMatching::AddModifySessionAttribute(const ModifySessionAttribute* attribute)
	{
		SimpleLock::AutoLock lock(m_Lock);

		m_LastResult.Reset();
		sce::Toolkit::NP::ModifySessionAttributes attrib;
		memset(&attrib, 0, sizeof(attrib));
		strncpy(attrib.attribute, attribute->name, SCE_TOOLKIT_NP_MAX_ATTRIBUTE_LENGTH);
		if(attribute->binValue && strlen(attribute->binValue) > 0)
		{
			strncpy(attrib.attributeValue.attributeBinValue, attribute->binValue, SCE_TOOLKIT_NP_ATTRIBUTE_MAX_BIN_VALUE);
		}
		else
		{
			attrib.attributeValue.attributeIntValue = attribute->intValue;
		}

		m_ModifyAttributes.push_back(attrib);
		return m_LastResult.GetResult();
	}

	ErrorCode CachedMatching::ModifySession(int attributeType)
	{
		if(SessionIsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		SimpleLock::AutoLock lock(m_Lock);

		m_LastResult.Reset();
		sce::Toolkit::NP::ModifySessionRequest modifyRequest;

		memset(&modifyRequest,0, sizeof(sce::Toolkit::NP::ModifySessionRequest));
		modifyRequest.attributeType = attributeType;
		modifyRequest.numAttributes = m_ModifyAttributes.size();
		modifyRequest.sessionAttributes = modifyRequest.numAttributes > 0 ? &m_ModifyAttributes[0] : NULL;

		int ret = Matching::Interface::modifySession(&modifyRequest, &m_FutureSessionResult);
		if (ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		m_SessionBusy = true;
		return m_LastResult.GetResult();
	}

	ErrorCode CachedMatching::LeaveSession()
	{
		if(SessionIsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		SimpleLock::AutoLock lock(m_Lock);

		m_LastResult.Reset();
		if(m_FutureSessionProcess.isBusy() || m_FutureSessionResult.isBusy()) 
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}

		int ret = Matching::Interface::leaveSession(&m_SessionInformation, &m_FutureSessionResult);
		if (ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		m_SessionBusy = true;
		return m_LastResult.GetResult();
	}

	ErrorCode CachedMatching::Search(int serverID, int worldID)
	{
		return SearchSession(serverID, worldID, 0);
	}

	ErrorCode CachedMatching::SearchFriends(int serverID, int worldID)
	{
		return SearchSession(serverID, worldID, SCE_TOOLKIT_NP_SEARCH_FRIENDS_SESSIONS);
	}

	ErrorCode CachedMatching::SearchRegional(int serverID, int worldID)
	{
		return SearchSession(serverID, worldID, SCE_TOOLKIT_NP_SEARCH_REGIONAL_SESSIONS);
	}

	ErrorCode CachedMatching::SearchSession(int serverID, int worldID, int searchFlags)
	{
		if(SessionIsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		SimpleLock::AutoLock lock(m_Lock);

		m_LastResult.Reset();

		ErrorCode result = CreateSessionSearchObject();
		if(result != NP_OK)
		{
			return m_LastResult.SetResult(result, true, __FUNCTION__, __LINE__);
		}

		m_SessionList.clear();

		SearchSessionsRequest searchCriteria;
		memset(&searchCriteria, 0, sizeof(SearchSessionsRequest));
#if __ORBIS__ || PSP2_USING_WEBAPI
		searchCriteria.userInfo.userId = GetUserId();
#endif
		searchCriteria.searchFilters = &m_SessionAttributes[0];
		searchCriteria.numSearchFilters =m_SessionAttributes.size(); 

		searchCriteria.serverId = serverID;
		searchCriteria.worldId = worldID;
		searchCriteria.searchFlags = searchFlags;

		unsigned int ret = Matching::Interface::searchSessions(&searchCriteria, m_FutureSearchProcess);
		if (ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			if (ret != SCE_TOOLKIT_NP_MATCHING_SERVICE_BUSY)
			{
				DestroySessionSearchObject();
			}
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		m_SessionBusy = true;
		return m_LastResult.GetResult();
	}

	ErrorCode CachedMatching::CreateSessionSearchObject()
	{
		if (m_FutureSearchProcess == NULL)
		{
			m_FutureSearchProcess = new Future<SessionList>;
			return NP_OK;
		}

		if(m_FutureSearchProcess->isBusy())
		{
			return NP_ERR_BUSY;
		}

		return NP_OK;
	}

	void CachedMatching::DestroySessionSearchObject()
	{
		if (m_FutureSearchProcess)
		{
			delete m_FutureSearchProcess;
			m_FutureSearchProcess = NULL;
		}
	}

	ErrorCode CachedMatching::GetSessionInfo(SessionInfo* info)
	{
		SimpleLock::AutoLock lock(m_Lock);
		m_LastResult.Reset();
		info->sessionName = m_SessionInformation.sessionName;
		info->sessionID = -1;
		info->maxMembers = m_SessionInformation.maxMembers;
		info->numMembers = m_SessionInformation.numMembers;
		info->numSessionAttributes = m_SessionInformation.numSessionAttributes;
		info->reservedSlots = m_SessionInformation.slotsInformation.reservedSlots;
		info->openSlots = m_SessionInformation.slotsInformation.openSlots;
		info->worldId = m_SessionInformation.worldId;
		info->serverId = m_SessionInformation.serverId;
		info->matchingContext = m_SessionInformation.matchingContext;
		info->roomId = m_SessionInformation.roomId;
		return m_LastResult.GetResult();
	}

	int CachedMatching::GetSessionAttributeCount()
	{
		return m_SessionInformation.sessionAttributes.size();
	}

	ErrorCode CachedMatching::GetSessionAttribute(unsigned int index, SessionAttributeInfo* info)
	{
		if(index >= m_SessionInformation.sessionAttributes.size())
		{
			return m_LastResult.SetResult(NP_ERR_INDEX_OUT_OF_RANGE, true, __FUNCTION__, __LINE__);
		}

		m_LastResult.Reset();

		sce::Toolkit::NP::SessionAttribute& attr = m_SessionInformation.sessionAttributes[index];

		info->attribute = attr.attribute;
		info->attributeType = attr.attributeType;
		info->searchOperator = attr.searchOperator;
		info->maxSize = attr.maxSize;
		info->attributeValueType = attr.attributeValueType;
		if(attr.attributeValueType == SCE_TOOLKIT_NP_SESSION_ATTRIBUTE_VALUE_INT)
		{
			info->attributeIntValue = attr.attributeValue.attributeIntValue;
			static const char* empty = "";
			info->attributeBinValue = empty;
		}
		else if(attr.attributeValueType == SCE_TOOLKIT_NP_SESSION_ATTRIBUTE_VALUE_BINARY)
		{
			info->attributeIntValue = 0;
			info->attributeBinValue = attr.attributeValue.attributeBinValue;
		}

		return m_LastResult.GetResult();
	}

	int CachedMatching::GetSessionMemberCount()
	{
		return m_SessionInformation.memberData.size();
	}

	ErrorCode CachedMatching::GetSessionMemberInfo(unsigned int index, SessionMemberInfo* info)
	{
		if(index >= m_SessionInformation.memberData.size())
		{
			return m_LastResult.SetResult(NP_ERR_INDEX_OUT_OF_RANGE, true, __FUNCTION__, __LINE__);
		}

		m_LastResult.Reset();

		sce::Toolkit::NP::SessionMember& member = m_SessionInformation.memberData[index];
		info->npOnlineID = (char*)member.userInfo.handle.data;
		info->npID = (unsigned char*)&member.userInfo;
		info->npIDSize = sizeof(SceNpId);
		info->memberId = member.memberId;
		info->natType = member.natType;
		info->memberFlag = member.memberFlag;
		info->joinDate = member.joinDate.tick;
		info->addr = member.memberConnInfo.addr.s_addr;
		info->port = member.memberConnInfo.port;

		return m_LastResult.GetResult();
	}

	int CachedMatching::GetSessionMemberAttributeCount(unsigned int memberIndex)
	{
		if(memberIndex >= m_SessionInformation.memberData.size())
		{
			m_LastResult.SetResult(NP_ERR_INDEX_OUT_OF_RANGE, true, __FUNCTION__, __LINE__);
			return 0;
		}

		sce::Toolkit::NP::SessionMember& member = m_SessionInformation.memberData[memberIndex];
		return member.memberAttributes.size();
	}

	ErrorCode CachedMatching::GetSessionMemberAttribute(unsigned int memberIndex, unsigned int index, SessionAttributeInfo* info)
	{
		if(memberIndex >= m_SessionInformation.memberData.size())
		{
			return m_LastResult.SetResult(NP_ERR_INDEX_OUT_OF_RANGE, true, __FUNCTION__, __LINE__);
		}

		sce::Toolkit::NP::SessionMember& member = m_SessionInformation.memberData[memberIndex];

		if(index >= member.memberAttributes.size())
		{
			return m_LastResult.SetResult(NP_ERR_INDEX_OUT_OF_RANGE, true, __FUNCTION__, __LINE__);
		}

		m_LastResult.Reset();

		sce::Toolkit::NP::SessionAttribute& attr = member.memberAttributes[index];

		info->attribute = attr.attribute;
		info->attributeType = attr.attributeType;
		info->searchOperator = attr.searchOperator;
		info->maxSize = attr.maxSize;
		info->attributeValueType = attr.attributeValueType;
		if(attr.attributeValueType == SCE_TOOLKIT_NP_SESSION_ATTRIBUTE_VALUE_INT)
		{
			info->attributeIntValue = attr.attributeValue.attributeIntValue;
			static const char* empty = "";
			info->attributeBinValue = empty;
		}
		else if(attr.attributeValueType == SCE_TOOLKIT_NP_SESSION_ATTRIBUTE_VALUE_BINARY)
		{
			info->attributeIntValue = 0;
			info->attributeBinValue = attr.attributeValue.attributeBinValue;
		}

		return m_LastResult.GetResult();
	}

	void CachedMatching::Lock()
	{
		m_Lock.Lock();
	}

	void CachedMatching::Unlock()
	{
		m_Lock.Unlock();
	}

	// Found session info.
	int CachedMatching::GetSessionFoundListCount()
	{
		return m_SessionList.size();
	}

	ErrorCode CachedMatching::GetSessionFoundInfo(unsigned int sessionIndex, SessionInfo* info)
	{
		if(sessionIndex >= m_SessionList.size())
		{
			return m_LastResult.SetResult(NP_ERR_INDEX_OUT_OF_RANGE, true, __FUNCTION__, __LINE__);
		}

		m_LastResult.Reset();

		sce::Toolkit::NP::SessionInformation& SessionInformation = m_SessionList[sessionIndex];
		info->sessionName = SessionInformation.sessionName;
		info->sessionID = sessionIndex;
		info->maxMembers = SessionInformation.maxMembers;
		info->numMembers = SessionInformation.numMembers;
		info->numSessionAttributes = SessionInformation.numSessionAttributes;
		info->reservedSlots = SessionInformation.slotsInformation.reservedSlots;
		info->openSlots = SessionInformation.slotsInformation.openSlots;
		info->worldId = SessionInformation.worldId;
		info->serverId = SessionInformation.serverId;
		info->matchingContext = SessionInformation.matchingContext;
		info->roomId = SessionInformation.roomId;

		return m_LastResult.GetResult();
	}

	int CachedMatching::GetSessionFoundAttributeCount(unsigned int sessionIndex)
	{
		if(sessionIndex >= m_SessionList.size())
		{
			m_LastResult.SetResult(NP_ERR_INDEX_OUT_OF_RANGE, true, __FUNCTION__, __LINE__);
			return 0;
		}

		return m_SessionList[sessionIndex].sessionAttributes.size();
	}

	ErrorCode CachedMatching::GetSessionFoundAttribute(unsigned int sessionIndex, unsigned int attributeIndex, SessionAttributeInfo* info)
	{
		if(sessionIndex >= m_SessionList.size())
		{
			return m_LastResult.SetResult(NP_ERR_INDEX_OUT_OF_RANGE, true, __FUNCTION__, __LINE__);
		}

		if(attributeIndex >= m_SessionList[sessionIndex].sessionAttributes.size())
		{
			return m_LastResult.SetResult(NP_ERR_INDEX_OUT_OF_RANGE, true, __FUNCTION__, __LINE__);
		}

		m_LastResult.Reset();

		sce::Toolkit::NP::SessionAttribute& attr = m_SessionList[sessionIndex].sessionAttributes[attributeIndex];

		info->attribute = attr.attribute;
		info->attributeType = attr.attributeType;
		info->searchOperator = attr.searchOperator;
		info->maxSize = attr.maxSize;
		info->attributeValueType = attr.attributeValueType;
		if(attr.attributeValueType == SCE_TOOLKIT_NP_SESSION_ATTRIBUTE_VALUE_INT)
		{
			info->attributeIntValue = attr.attributeValue.attributeIntValue;
			static const char* empty = "";
			info->attributeBinValue = empty;
		}
		else if(attr.attributeValueType == SCE_TOOLKIT_NP_SESSION_ATTRIBUTE_VALUE_BINARY)
		{
			info->attributeIntValue = 0;
			info->attributeBinValue = attr.attributeValue.attributeBinValue;
		}

		return m_LastResult.GetResult();
	}

	// Session invites
	ErrorCode CachedMatching::InviteToSession(const char* text, int npIdsCount)
	{
		m_LastResult.Reset();

		InviteMessage msg;
		INIT(msg);

		msg.body = text;


#if __ORBIS__ || PSP2_USING_WEBAPI
		msg.dialogFlag = SCE_TOOLKIT_NP_DIALOG_TYPE_USER_EDITABLE;
		msg.npIdsCount = npIdsCount;
#endif


		int ret = Matching::Interface::inviteToSession(&m_SessionInformation, &msg);
		if (ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		return m_LastResult.GetResult();
	}





}; //namespace UnityPlugin
